<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreTodoItemRequest;
use App\Http\Requests\UpdateTodoItemRequest;
use App\Models\TodoItem;
use App\Repositories\TodoItemRepository;
use Exception;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Response;

/**
 * Class TodoItemController
 * @package App\Http\Controllers
 */
class TodoItemController extends Controller
{
    /**
     * @var TodoItemRepository
     */
    protected TodoItemRepository $todoItemRepository;

    /**
     * @param TodoItemRepository $todoItemRepository
     */
    public function __construct(TodoItemRepository $todoItemRepository)
    {
        $this->todoItemRepository = $todoItemRepository;
    }

    /**
     * Display a listing of the resource.
     *
     * @return View
     */
    public function index(): View
    {
        return view('todo-items.index');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @param TodoItem $todoItem
     * @return Application|Factory|View
     */
    public function show(TodoItem $todoItem)
    {
        return view('todo-items.show', compact('todoItem'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return View
     */
    public function create(): View
    {
        return view('todo-items.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param StoreTodoItemRequest $request
     * @return RedirectResponse
     */
    public function store(StoreTodoItemRequest $request): RedirectResponse
    {
        $this->todoItemRepository->create($request->validated());

        return redirect()->route('todo-items.index');
    }

    /**
     * Update the specified resource in storage.
     *
     * @param UpdateTodoItemRequest $request
     * @param TodoItem $todoItem
     * @return JsonResponse
     */
    public function update(UpdateTodoItemRequest $request, TodoItem $todoItem): JsonResponse
    {
        $this->todoItemRepository->updateOrCreate(['id' => $todoItem->id], $request->validated());

        return response()->json(null, Response::HTTP_ACCEPTED);
    }

    /**
     * @param TodoItem $todoItem
     * @return JsonResponse
     * @throws Exception
     */
    public function destroy(TodoItem $todoItem): JsonResponse
    {
        $this->todoItemRepository->delete($todoItem);

        return response()->json([], Response::HTTP_NO_CONTENT);
    }
}
