<?php

namespace App\Repositories;

use App\Enums\OrderType;
use Carbon\Carbon;
use Exception;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Contracts\Pagination\Paginator;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;

/**
 * Class BaseRepository
 * @package App\Repositories
 */
abstract class BaseRepository
{
    /**
     * The model instance.
     *
     * @var Model
     */
    protected Model $model;

    /**
     * Create a new model instance.
     *
     * @param Model $model
     */
    public function __construct(Model $model)
    {
        $this->model = $model;
    }

    /**
     * Returns the first record in the database.
     *
     * @return Model
     */
    public function first(): Model
    {
        return $this->model->first();
    }

    /**
     * Returns all the records.
     *
     * @return Collection
     */
    public function all(): Collection
    {
        return $this->model->all();
    }

    /**
     * Returns the count of all the records.
     *
     * @return int
     */
    public function count(): int
    {
        return $this->model->count();
    }

    /**
     * Returns a range of records bounded by pagination parameters.
     *
     * @param int $limit
     * @param int $offset
     * @param array $relations
     * @param string $orderBy
     * @param string $sorting
     * @return Collection
     */
    public function page(
        int $limit = 10,
        int $offset = 0,
        array $relations = [],
        string $orderBy = 'updated_at',
        string $sorting = 'desc'
    ): Collection {
        return $this->model->with($relations)->take($limit)->skip($offset)->orderBy($orderBy, $sorting)->get();
    }

    /**
     * Find a record by its identifier.
     *
     * @param string $id
     * @param array|null $relations
     *
     * @return Model
     */
    public function find(string $id, array $relations = null): Model
    {
        return $this->findBy($this->model->getKeyName(), $id, $relations);
    }

    /**
     * Increment a record by its column
     *
     * @param Model $model
     * @param string $column
     *
     * @return void
     */
    public function increment(Model $model, string $column = 'total_shows')
    {
        $model->increment($column);
    }

    /**
     * Find a record by an attribute.
     * Fails if no model is found.
     *
     * @param string $attribute
     * @param string $value
     * @param array|null $relations
     * @param bool $canFail
     *
     * @return Model|null
     */
    public function findBy(
        string $attribute,
        string $value,
        array $relations = null,
        bool $canFail = true
    ): ?Model {
        $query = $this->model->where($attribute, $value);

        if ($relations && is_array($relations)) {
            foreach ($relations as $relation) {
                $query->with($relation);
            }
        }

        return $query->firstOrFail();
    }

    /**
     * Get model where column in array.
     *
     * @param array $attributes
     * @param array $relations
     * @param string $column
     * @param array $properties
     * @param int $limit
     * @return Collection
     */
    public function whereIn(
        array $attributes,
        array $relations = [],
        string $column = 'id',
        array $properties = ['*'],
        int $limit = 50
    ): Collection {
        return $this
            ->model
            ->with($relations)
            ->whereIn($column, $attributes)
            ->limit($limit)
            ->get($properties);
    }

    /**
     * Paginate records where column in array.
     *
     * @param string $column
     * @param array $attributes
     * @param array $properties
     * @param int $limit
     * @param array $relations
     * @param string $orderType
     * @param string $orderColumn
     *
     * @return LengthAwarePaginator|Paginator
     */
    public function paginateWhereIn(
        string $column = 'id',
        array $attributes,
        array $properties = ['*'],
        int $limit = 10,
        array $relations = [],
        string $orderType = OrderType::DESC,
        string $orderColumn = 'created_at'
    ): LengthAwarePaginator {
        return $this
            ->model
            ->with($relations)
            ->whereIn($column, $attributes)
            ->orderBy($orderColumn, $orderType)
            ->paginate($limit, $properties);
    }

    /**
     * Get count of all records.
     *
     * @param array $attributes
     * @param string $dateColumn
     * @param Carbon $date
     *
     * @return int
     */
    public function countByDate(
        array $attributes,
        Carbon $date = null,
        string $dateColumn = 'created_at'
    ): int {
        $query = $this
            ->model
            ->where($attributes);

        if ($date) {
            $query->whereDate($dateColumn, Carbon::parse($date)->format('Y-m-d'));
        }

        return $query->count();
    }

    /**
     * Get all records by an associative array of attributes.
     * Two operators values are handled: AND | OR.
     *
     * @param array $attributes
     * @param string $operator
     * @param array|null $relations
     *
     * @return Collection
     */
    public function getByAttributes(
        array $attributes,
        string $operator = 'AND',
        array $relations = null
    ): Collection {
        // In the following it doesn't matter wivh element to start with, in all cases all attributes will be appended to the
        // builder.

        // Get the last value of the associative array
        $lastValue = end($attributes);

        // Get the last key of the associative array
        $lastKey = key($attributes);

        // Builder
        $query = $this->model->where($lastKey, $lastValue);

        // Pop the last key value pair of the associative array now that it has been added to Builder already
        array_pop($attributes);

        $method = 'where';

        if (strtoupper($operator) === 'OR') {
            $method = 'orWhere';
        }

        foreach ($attributes as $key => $value) {
            $query->$method($key, $value);
        }

        if ($relations && is_array($relations)) {
            foreach ($relations as $relation) {
                $query->with($relation);
            }
        }

        return $query->get();
    }

    /**
     * Get first record by an associative array of attributes.
     * Two operators values are handled: AND | OR.
     *
     * @param array $attributes
     * @param string $operator
     * @param array|null $relations
     *
     * @return Model|null
     */
    public function getFirstByAttributes(array $attributes,string  $operator = 'AND',array $relations = null): ?Model
    {
        // In the following it doesn't matter wivh element to start with, in all cases all attributes will be appended to the
        // builder.

        // Get the last value of the associative array
        $lastValue = end($attributes);

        // Get the last key of the associative array
        $lastKey = key($attributes);

        // Builder
        $query = $this->model->where($lastKey, $lastValue);

        // Pop the last key value pair of the associative array now that it has been added to Builder already
        array_pop($attributes);

        $method = 'where';

        if (strtoupper($operator) === 'OR') {
            $method = 'orWhere';
        }

        foreach ($attributes as $key => $value) {
            $query->$method($key, $value);
        }

        if ($relations && is_array($relations)) {
            foreach ($relations as $relation) {
                $query->with($relation);
            }
        }

        return $query->firstOrFail();
    }

    /**
     * Fills out an instance of the model
     * with $attributes.
     *
     * @param array $attributes
     *
     * @return Model
     */
    public function fill(array $attributes): Model
    {
        return $this->model->fill($attributes);
    }

    /**
     * Fills out an instance of the model
     * and saves it, pretty much like mass assignment.
     *
     * @param array $attributes
     *
     * @return Model
     */
    public function fillAndSave(array $attributes): Model
    {
        $this->model->fill($attributes);
        $this->model->save();

        return $this->model;
    }

    /**
     * Remove a selected record.
     *
     * @param mixed $key
     *
     * @return bool
     */
    public function remove($key): bool
    {
        return $this->model->where($this->model->getKeyName(), $key)->delete();
    }

    /**
     * Implement a convenience call to findBy
     * which allows finding by an attribute name
     * as follows: findByName or findByAlias.
     *
     * @param string $method
     * @param array $arguments
     *
     * @return mixed
     */
    public function __call(string $method, array $arguments)
    {
        /*
         * findBy convenience calling to be available
         * through findByName and findByTitle etc.
         */

        if (preg_match('/^findBy/', $method)) {
            $attribute = strtolower(substr($method, 6));
            array_unshift($arguments, $attribute);

            return call_user_func_array(array($this, 'findBy'), $arguments);
        }
    }


    /**
     * Search models by attribute
     *
     * @param array $attributes
     * @param string $searchValue
     * @param string $operator
     * @param bool $isStrict
     * @param array $relations
     * @return mixed
     */
    public function searchByAttributes(
        array $attributes,
        string $searchValue,
        string $operator = 'AND',
        bool $isStrict = true,
        array $relations = []
    ) {
        $lastAttribute = end($attributes);
        $query = $this->model->where($lastAttribute, $searchValue);

        $method = 'where';

        if (strtoupper($operator) === 'OR') {
            $method = 'orWhere';
        }

        if (!$isStrict) {
            $searchValue = "%{$searchValue}%";
        }

        foreach ($attributes as $attribute) {
            $query->$method($attribute, 'LIKE', $searchValue);
        }

        return $query->with($relations)->get();
    }

    /**
     * Retrieve paginated records
     *
     * @param int $limit
     * @param array $relations
     * @param bool $isSimplePagination
     * @param string $orderBy
     * @param string $orderColumn
     *
     * @return LengthAwarePaginator|Paginator
     */
    public function getAllPaginatedResults(
        int $limit = 5,
        array $relations = [],
        bool $isSimplePagination = false,
        string $orderBy = OrderType::DESC,
        string $orderColumn = 'created_at'
    ) {
        $builder = $this
            ->model
            ->with($relations)
            ->orderBy($orderColumn, $orderBy);

        return $isSimplePagination ? $builder->simplePaginate($limit) : $builder->paginate($limit);
    }

    /**
     * Check if model exists
     *
     * @param array $attributes
     * @param bool $withDeleted
     * @return bool
     */
    public function existsByAttributes(array $attributes, bool $withDeleted = false): bool
    {
        $query = $this->model;

        if ($withDeleted) {
            $query->withTrashed();
        }

        return $query
            ->where($attributes)
            ->exists();
    }

    /**
     * Paginates by attributes.
     *
     * @param array $attributes
     * @param int $limit
     * @param string $order
     * @return LengthAwarePaginator
     */
    public function paginateByAttributes(
        array $attributes = [],
        int $limit = 10,
        string $order = OrderType::DESC
    ): LengthAwarePaginator {
        $query = $this->model;

        if (!empty($attributes)) {
            foreach ($attributes as $key => $attribute) {
                $query = $query->where($key, '=', $attribute);
            }
        }

        return $query
            ->orderBy('created_at', $order)
            ->paginate($limit);
    }

    /**
     * Update model by id.
     *
     * @param int|string $id
     * @param array $data
     * @return bool
     */
    public function update(int $id, array $data): bool
    {
        return $this
            ->findBy($this->model->getKeyName(), $id)
            ->update($data);
    }

    /**
     * Update model by key/value pair.
     *
     * @param string $column
     * @param mixed $value
     * @param array $data
     * @return bool
     */
    public function updateBy(string $column, $value, array $data): bool
    {
        return $this
            ->model
            ->where($column, $value)
            ->update($data);
    }

    /**
     * Get model from soft deletes.
     *
     * @param int $id
     * @param array $relations
     * @return Model
     */
    public function findInSoftDeletes(int $id, array $relations = []): Model
    {
        return $this
            ->model
            ->onlyTrashed()
            ->with($relations)
            ->findOrFail($id);
    }

    /**
     * Restore soft deleted model.
     *
     * @param int $id
     * @return bool
     */
    public function restoreSoftDeletedRecord(int $id): bool
    {
        return $this->findInSoftDeletes($id)->restore();
    }

    /**
     * Delete records by key/value pair.
     *
     * @param string $key
     * @param mixed $value
     * @return bool
     */
    public function deleteByKey(string $key, $value): bool
    {
        return $this
            ->model
            ->where($key, $value)
            ->delete();
    }

    /**
     * Force delete records by key/value pair.
     *
     * @param string $key
     * @param mixed $value
     * @return bool
     */
    public function forceDeleteByKey(string $key, $value): bool
    {
        return $this
            ->model
            ->where($key, $value)
            ->forceDelete();
    }

    /**
     * Retrieves user associated with .
     *
     * @param string $attribute
     * @param $value
     * @return mixed
     */
    public function getAssociatedUserByAttribute(string $attribute, $value)
    {
        $searchColumn = "{$this->model->getTable()}.{$attribute}";

        return $this
            ->model
            ->select('users.*')
            ->join('users', 'users.id', '=', "{$this->model->getTable()}.user_id")
            ->where($searchColumn, $value)
            ->first();
    }

    /**
     * Deletes records by attributes.
     *
     * @param array $attributes
     * @param bool $isForceDelete
     * @return bool
     */
    public function deleteByAttributes(array $attributes, bool $isForceDelete = false): bool
    {
        $query = $this
            ->model
            ->where($attributes);

        return $isForceDelete ? $query->forceDelete() : $query->delete();
    }

    /**
     * Get first record by attributes.
     *
     * @param array $attributes
     * @return mixed
     */
    public function firstByAttributes(array $attributes)
    {
        return $this
            ->model
            ->where($attributes)
            ->first();
    }

    /**
     * Count models by attributes.
     *
     * @param array $attributes
     * @return mixed
     */
    public function countByAttributes(array $attributes): int
    {
        return $this
            ->model
            ->where($attributes)
            ->count();
    }

    /**
     * Delete model
     *
     * @param Model $model
     * @param bool $isForceDelete
     * @return bool|null
     * @throws Exception
     */
    public function delete(Model $model, bool $isForceDelete = false): ?bool
    {
        return $isForceDelete ? $model->forceDelete() : $model->delete();
    }

    /**
     * Pluck records.
     *
     * @param string $column
     * @return mixed
     */
    public function pluck(string $column)
    {
        return $this
            ->model
            ->pluck($this->model->getKeyName(), $column);
    }

    /**
     * Update or create model.
     *
     * @param array $conditions
     * @param array $data
     * @return mixed
     */
    public function updateOrCreate(array $conditions, array $data)
    {
        return $this->model->updateOrCreate($conditions, $data);
    }

    /**
     * create model.
     *
     * @param array $conditions
     * @param array $data
     * @return mixed
     */
    public function create(array $attributes)
    {
        return $this->model->create($attributes);
    }

    /**
     * Retrieve last model by attributes
     *
     * @param array $attributes
     * @return mixed
     */
    public function lastByAttributes(array $attributes)
    {
        return $this
            ->model
            ->where($attributes)
            ->latest()
            ->first();
    }
}
