<?php

namespace App\Repositories;

use Illuminate\Database\Eloquent\Model;

/**
 * Class Repository
 * @package App\Repositories
 */
abstract class Repository
{
    /**
     * @var \Illuminate\Database\Eloquent\Model
     */
    protected $model;

    /**
     * @return string
     */
    abstract protected function getModelClass(): string;

    /**
     * Repository constructor.
     */
    public function __construct()
    {
        $model = app($this->getModelClass());
        if (!$model instanceof Model) {
            throw new \InvalidArgumentException(
                'Class "' . $this->getModelClass() . '" should be an instance of ' . Model::class
            );
        }
        $this->model = $model;
    }

    /**
     * Find all items.
     *
     * @param array $columns
     * @return \Illuminate\Database\Eloquent\Builder[]|\Illuminate\Database\Eloquent\Collection
     */
    public function all(array $columns = ['*'])
    {
        return $this->model::query()->get($columns);
    }

    /**
     * Find one item by primary key.
     *
     * @param $id
     * @param array $columns
     * @return \Illuminate\Database\Eloquent\Collection|\Illuminate\Database\Eloquent\Model|null
     */
    public function find($id, array $columns = ['*'])
    {
        return $this->model::query()->find($id, $columns);
    }

    /**
     * Save item in database.
     *
     * @param array $attributes
     * @param array $translations
     *
     * @return \Illuminate\Database\Eloquent\Builder|\Illuminate\Database\Eloquent\Model
     */
    public function store(array $attributes = [], array $translations = [])
    {
        $result = $this->model::query()->create($attributes);
        if ($translations) {
            $result->storeTranslations($translations);
        }

        return $result;
    }

    /**
     * Save or update item in database.
     *
     * @param array $find
     * @param array $attributes
     * @param array $translations
     * @return \Illuminate\Database\Eloquent\Builder|\Illuminate\Database\Eloquent\Model
     */
    public function updateOrStore(array $find = [], array $attributes = [], array $translations = [])
    {
        $result = $this->model::query()->updateOrCreate($find, $attributes);
        if ($translations) {
            $result->storeTranslations($translations);
        }

        return $result;
    }


}
