<?php

namespace App\Models\Regions;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use KoenHoeijmakers\LaravelTranslatable\HasTranslations;
use Spatie\MediaLibrary\HasMedia\HasMedia;
use Spatie\MediaLibrary\HasMedia\HasMediaTrait;
use Spatie\MediaLibrary\Models\Media;

/**
 * Class City
 * @package App\Models\Regions
 *
 * @property int $id
 * @property string $alias
 * @property string $name
 * @property string $mayor
 * @property string $address
 * @property string $phone
 * @property string $fax
 * @property string $email
 * @property string $web
 * @property string $geo
 *
 * @property Collection $phones
 * @property Collection $faxes
 * @property Collection $emails
 * @property Collection $sites
 * @property Collection $location
 *
 * @property \App\Models\Regions\District $district
 * @property \Spatie\MediaLibrary\Models\Media $emblem
 */
class City extends Model implements HasMedia
{
    use HasTranslations, HasMediaTrait;

    public const COLLECTION_EMBLEM = 'emblem';

    /**
     * {@inheritdoc}
     */
    protected $translatable = ['name'];

    /**
     * {@inheritdoc}
     */
    protected $fillable = ['name', 'district_id', 'alias', 'mayor', 'address', 'phones', 'faxes', 'emails', 'sites',
        'coordinates'];

    /**
     * @var array Allowed mime types
     */
    protected $allowedImageMimeTypes = ['image/gif', 'image/jpeg'];

    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function district(): BelongsTo
    {
        return $this->belongsTo(District::class);
    }

    /**
     * @param string $url
     */
    public function updateEmblem(string $url): void
    {
        try {
            $this->addMediaFromUrl($url, $this->allowedImageMimeTypes)
                ->toMediaCollection(static::COLLECTION_EMBLEM);
        } catch (\Throwable $exception) {
            Log::warning('Cannot update emblem.');
        }
    }

    /**
     * @return \Spatie\MediaLibrary\Models\Media|null $media
     */
    public function getEmblemAttribute(): ?Media
    {
        return $this->getMedia(static::COLLECTION_EMBLEM)->last();
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function getPhonesAttribute(): Collection
    {
        return $this->getMultiProps('phone');
    }

    /**
     * @param $values
     */
    public function setPhonesAttribute($values): void
    {
        $this->setMultiProps('phone', $values);
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function getFaxesAttribute(): ?Collection
    {
        return $this->getMultiProps('fax');
    }

    /**
     * @param $values
     */
    public function setFaxesAttribute($values): void
    {
        $this->setMultiProps('fax', $values);
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function getEmailsAttribute(): ?Collection
    {
        return $this->getMultiProps('email');
    }

    /**
     * @param $values
     */
    public function setEmailsAttribute($values): void
    {
        $this->setMultiProps('email', $values);
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function getSitesAttribute(): ?Collection
    {
        return $this->getMultiProps('web');
    }

    /**
     * @param $values
     */
    public function setSitesAttribute($values): void
    {
        $this->setMultiProps('web', $values);
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function getCoordinatesAttribute(): ?Collection
    {
        return (!$this->geo) ? null : collect(json_decode($this->geo, true));
    }

    /**
     * @param $values
     */
    public function setCoordinatesAttribute($values): void
    {
        if (($geo = collect($values)) && $geo->has('lat') && $geo->has('lng')) {
            $this->geo = json_encode(['lat' => $geo->get('lat'), 'lng' => $geo->get('lng')]);
        }
    }

    /**
     * Set property as array
     *
     * @param string $attribute
     * @param mixed $values
     */
    protected function setMultiProps($attribute, $values): void
    {
        $this->{$attribute} = collect($values)->map('trim')->filter()->values()->toJson();
    }

    /**
     * Get property as array
     *
     * @param string $attribute
     * @return \Illuminate\Support\Collection
     */
    protected function getMultiProps(string $attribute): Collection
    {
        $data = $this->{$attribute};
        try {
            $data = json_decode($data, true);
        } catch (\Throwable $exception) {
            Log::warning('Unable to decode "' . __CLASS__ . ':' . $attribute . '"');
        }
        return collect($data)->map('trim')->values();
    }
}
