<?php

namespace App\Services\Regions;

use App\Contracts\CityContract;
use App\Contracts\DistrictContract;
use App\Contracts\RegionContract;
use App\Contracts\RegionParserContract;
use App\Models\Regions\District;
use App\Services\Regions\Components\Client;
use App\Services\Regions\Components\Converter;
use App\Services\Regions\Components\Parser;
use Illuminate\Support\Collection;

/**
 * Class RegionParseService
 * @package App\Services\Regions
 */
class RegionParseService implements RegionParserContract
{
    /**
     * Base site URL.
     */
    public const BASE_URL = 'https://www.e-obce.sk';

    /**
     * @var \App\Services\Regions\Components\Client HTTP-client.
     */
    protected $client;

    /**
     * @var \App\Services\Regions\Components\Converter
     */
    protected $converter;
    /**
     * @var \App\Services\Regions\Components\Parser
     */
    protected $parser;

    /**
     * @var \App\Contracts\RegionContract
     */
    protected $regions;

    /**
     * @var \App\Contracts\DistrictContract
     */
    protected $districts;

    /**
     * @var \App\Contracts\CityContract
     */
    protected $cities;

    /**
     * ParseRegionService constructor.
     *
     * @param \App\Contracts\RegionContract $regions
     * @param \App\Contracts\DistrictContract $districts
     * @param \App\Contracts\CityContract $cities
     */
    public function __construct(RegionContract $regions, DistrictContract $districts, CityContract $cities)
    {
        $this->regions = $regions;
        $this->districts = $districts;
        $this->cities = $cities;

        $this->client = new Client(static::BASE_URL);
        $this->converter = new Converter();
        $this->parser = new Parser();
    }

    /**
     * Get all regions available.
     *
     * @return \Illuminate\Support\Collection
     */
    public function getRegions()
    {
        return $this->regions->all();
    }

    /**
     * Parse region.
     *
     * @param \App\Models\Regions\Region $region
     * @return \Illuminate\Support\Collection
     */
    public function parseRegion($region): Collection
    {
        $result = collect();
        $response = $this->client->get($this->client->makeRegionUrl($region->alias));

        $this->parser->parseRegion($response)
            ->each(function ($item, $url) use ($result, $region) {

                [$name, $alias, $converted] = $item;

                $result->put($url, $this->districts->updateOrStore(
                    ['region_id' => $region->id, 'alias' => $alias],
                    ['name' => $converted, 'region_id' => $region->id, 'alias' => $alias],
                    ['sk' => ['name' => $name]]
                ));
            });

        return $result;
    }

    /**
     * Parse cities.
     *
     * @param string $url
     * @param District $district
     * @return \Illuminate\Support\Collection|void
     */
    public function parseDistrict(string $url, $district)
    {
        $result = collect();
        $response = $this->client->get($url);

        $this->parser->parseDistrict($response)
            ->each(function ($item, $parseUrl) use ($result, $district) {

                [$name, $alias, $converted] = $item;

                $result->put($parseUrl, $this->cities->updateOrStore(
                    ['district_id' => $district->id, 'alias' => $alias],
                    ['name' => $converted, 'district_id' => $district->id, 'alias' => $alias],
                    ['sk' => ['name' => $name]]
                ));
            });

        return $result;
    }

    /**
     * Parse city.
     *
     * @param string $url
     * @param $city
     */
    public function parseCity(string $url, $city)
    {
        $response = $this->client->get($url);

        if ($data = $this->parser->parseCity($response)) {
            $this->cities->updateContacts($city->id, $data);
        }
    }
}
