<?php


namespace App\Services\Regions\Components;

use Illuminate\Support\Collection;
use Symfony\Component\DomCrawler\Crawler;

/**
 * Class Parser
 * @package App\Services\Regions\Components
 */
class Parser
{
    /**
     * @var array Additional fields relations.
     */
    private static $additional = [
        'Starosta' => 'mayor',
        'Primátor' => 'mayor',
    ];

    /**
     * @param \Symfony\Component\DomCrawler\Crawler $html
     * @return \Illuminate\Support\Collection
     */
    public function parseRegion(Crawler $html): Collection
    {
        $result = collect();
        $html->filter('td[colspan="2"][valign="top"] > a')->each(function (Crawler $node) use ($result) {
            $url = $node->attr('href');
            $result->put($url, $this->collectNames($node));
        });
        return $result;
    }

    /**
     * @param \Symfony\Component\DomCrawler\Crawler $html
     * @return \Illuminate\Support\Collection
     */
    public function parseDistrict(Crawler $html): Collection
    {
        $result = collect();
        $html->filter('table[cellspacing="3"] td[valign="top"] > a')->each(function (Crawler $node) use ($result) {
            $url = $node->attr('href');
            $result->put($url, $this->collectNames($node));
        });
        return $result;
    }

    /**
     * @param \Symfony\Component\DomCrawler\Crawler $html
     * @return \Illuminate\Support\Collection
     */
    public function parseCity(Crawler $html): Collection
    {
        $result = collect();
        $tables = $html->filter('table[cellspacing="3"]');

        $tables->eq(0)->each(function (Crawler $node) use ($result) {
            if ($emblem = $this->findCityEmblem($node)) {
                $result->put('image', $emblem);
            }
            $this->collectContacts($node, $result);

            $address = collect();
            $node->filter('td[valign="top"]')->each(static function (Crawler $item, $index) use ($address) {
                if ($index <= 3) {
                    return;
                }

                if ($line = trim($item->text())) {
                    $address->push($line);
                }
            });

            $result->put('address', $address->implode(', '));
        });

        $tables->eq(1)->filter('tr')->each(function (Crawler $node) use ($result) {
            $cells = $node->filter('td');
            $title = preg_replace('|:$|u', '', $cells->eq(0)->text());
            $value = $cells->eq(1)->text();

            if ($field = $this->hasAdditional(trim($title))) {
                $result->put($field, $value);
            }
        });

        return $result;
    }

    /**
     * @param string $title
     * @return string|null
     */
    private function hasAdditional(string $title): ?string
    {
        return static::$additional[$title] ?? null;
    }

    /**
     * Collect contacts data.
     *
     * @param \Symfony\Component\DomCrawler\Crawler $node
     * @param \Illuminate\Support\Collection $result
     */
    private function collectContacts(Crawler $node, Collection $result): void
    {
        $node->filter('tr')->each(function (Crawler $item, $index) use ($result) {
            if ($index < 3) {
                return;
            }
            switch ($index) {
                case 3:
                    $result->put('phones',
                        $this->preparePhones(
                            $this->collectPhone($item)
                        )
                    );
                    break;
                case 4:
                    return;
                case 5:
                    $result->put(
                        'faxes',
                        $this->preparePhones(
                            $this->collectFax($item)
                        )
                    );
                    break;
                case 6:
                    $result->put('emails', $this->collectLinks($item, 'mailto:'));
                    break;
                case 7:
                    $result->put('sites', $this->collectLinks($item));
                    break;
            }
        });
    }

    /**
     * Collect links.
     *
     * @param \Symfony\Component\DomCrawler\Crawler $node
     * @param string|null $replace
     * @return array
     */
    private function collectLinks(Crawler $node, ?string $replace = null): array
    {
        $result = collect();
        $node->filter('a')->each(static function (Crawler $item) use ($result, $replace) {
            $result->push($replace ? str_replace($replace, '', $item->attr('href')) : $item->attr('href'));
        });

        return $result->toArray();
    }

    /**
     * Collect phone numbers.
     *
     * @param \Symfony\Component\DomCrawler\Crawler $node
     * @return Collection
     */
    private function collectFax(Crawler $node): Collection
    {
        return collect(explode(PHP_EOL, $node->filter('td')->last()->text()));
    }

    /**
     * Collect phone numbers.
     *
     * @param \Symfony\Component\DomCrawler\Crawler $node
     * @return Collection
     */
    private function collectPhone(Crawler $node): Collection
    {
        $phones = collect();
        $node->filter('table tr td')->each(static function (Crawler $item) use ($phones) {
            $phones->push($item->text());
        });
        return $phones;
    }

    /**
     * Look at city emblem.
     *
     * @param \Symfony\Component\DomCrawler\Crawler $node
     * @return string|null
     */
    private function findCityEmblem(Crawler $node): ?string
    {
        return ($img = $node->filter('td[rowspan="4"] img')->first()) ? $img->attr('src') : null;
    }

    /**
     * @param Crawler $node
     * @return array
     */
    private function collectNames(Crawler $node): array
    {
        $name = $node->text();
        $converted = $this->convertName($name);
        $alias = $this->createUrlAlias($converted);
        return [$name, $alias, $converted];
    }

    /**
     * Generate URL alias.
     *
     * @param string $name
     * @return string|null
     */
    private function createUrlAlias(string $name): ?string
    {
        $name = strtolower($name);
        $name = preg_replace('|[^a-z ]|u', '', $name);
        $name = preg_replace('|[\s]{2,}|u', ' ', $name);
        return preg_replace('|[\s]|u', '-', trim($name));
    }

    /**
     * Convert name to ASCII (for semi-automated English translation)
     *
     * @param string $name
     * @return string
     */
    private function convertName(string $name): string
    {
        return str_replace(['\''], [''], iconv('UTF-8', 'ASCII//TRANSLIT', $name));
    }

    /**
     * Prepare phone numbers.
     *
     * @param \Illuminate\Support\Collection $phones
     * @return array
     */
    private function preparePhones(Collection $phones): array
    {
        $result = collect();

        $phones->each(function (string $phone) use (&$result) {
            if ($duplicates = $this->duplicatePhone($phone)) {
                $result = $result->merge($duplicates);
            } else {
                $result->push($phone);
            }
        });

        return $result->toArray();
    }

    /**
     * Duplicate phone numbers.
     *
     * @param string $phone
     * @return \Illuminate\Support\Collection|null
     */
    private function duplicatePhone(string $phone): ?Collection
    {
        if (false === strpos($phone, ',')) {
            return null;
        }

        $parts = collect(explode(',', $phone))->map(static function (string $item) {
            return trim($item);
        });
        $canonical = $parts->shift();

        return $parts->map(static function (string $item) use ($canonical) {
            $length = -mb_strlen($item);
            return mb_substr($canonical, 0, $length) . $item;
        })->push($canonical);
    }
}
